# pong_singleplayer.py
# Pong Einzelspieler mit KI – Spieler links mit Pfeiltasten

from kivy.app import App
from kivy.uix.widget import Widget
from kivy.properties import NumericProperty, ReferenceListProperty, ObjectProperty
from kivy.vector import Vector
from kivy.clock import Clock
from random import randint, random
from kivy.core.window import Window

# --- Spielobjekte ---
class PongBall(Widget):
    velocity_x = NumericProperty(0)
    velocity_y = NumericProperty(0)
    velocity = ReferenceListProperty(velocity_x, velocity_y)
    
    def move(self):
        self.pos = Vector(*self.velocity) + self.pos

class PongPaddle(Widget):
    score = NumericProperty(0)
    
    def bounce_ball(self, ball):
        if self.collide_widget(ball):
            vx, vy = ball.velocity
            offset = (ball.center_y - self.center_y) / (self.height / 2)
            bounced = Vector(-1*vx, vy)
            ball.velocity = bounced.x*1.1, (bounced.y+offset)*1.1

class PongGame(Widget):
    ball = ObjectProperty(None)
    player1 = ObjectProperty(None)
    player2 = ObjectProperty(None)
    speed_timer = 0
    
    def serve_ball(self, vel=(4,0)):
        self.ball.center = self.center
        self.ball.velocity = Vector(*vel).rotate(randint(0,360))
        self.speed_timer = 0

    def update(self, dt):
        self.ball.move()
        self.speed_timer += dt

        # Ball beschleunigt alle 5 Sekunden
        if self.speed_timer > 5:
            self.ball.velocity = (self.ball.velocity_x*1.1, self.ball.velocity_y*1.1)
            self.speed_timer = 0

        # Ball prallt oben/unten ab
        if self.ball.y < 0 or self.ball.top > self.height:
            self.ball.velocity_y *= -1

        # Spieler-Schläger
        self.player1.bounce_ball(self.ball)
        self.player2.bounce_ball(self.ball)

        # KI-Bewegung für rechten Spieler
        self.ai_move(dt)

        # Punkte
        if self.ball.x < self.x:
            self.player2.score += 1
            self.serve_ball((4,0))
        if self.ball.right > self.width:
            self.player1.score += 1
            self.serve_ball((-4,0))

    def ai_move(self, dt):
        """Einfache KI für rechten Spieler mit gelegentlicher Fehlerchance"""
        error = random() < 0.1  # 10% Fehlerchance
        if self.ball.center_y > self.player2.center_y and not error:
            self.player2.y += min(10, self.ball.center_y - self.player2.center_y)
        elif self.ball.center_y < self.player2.center_y and not error:
            self.player2.y -= min(10, self.player2.center_y - self.ball.center_y)
        # Fehler: KI bleibt kurz stehen

    def keyboard_action(self, keycode):
        """Tastatursteuerung für linken Spieler mit Pfeiltasten"""
#        if keycode[1] == 'up':
#            self.player1.y += 30
#        elif keycode[1] == 'down':
#            self.player1.y -= 30

# --- App-Klasse ---
class PongApp(App):
    kv_file = None  # verhindert, dass Kivy versucht, externe .kv Datei zu laden

    def build(self):
        game = PongGame()
        from kivy.lang import Builder
        Builder.load_string("""
<PongBall>:
    size: 25,25
    canvas:
        Color:
            rgba: 1,1,1,1
        Ellipse:
            pos: self.pos
            size: self.size

<PongPaddle>:
    size: 25,200
    canvas:
        Color:
            rgba: 1,1,1,1
        Rectangle:
            pos: self.pos
            size: self.size

<PongGame>:
    ball: pong_ball
    player1: left_paddle
    player2: right_paddle
    canvas.before:
        Color:
            rgba:0,0,0,1
        Rectangle:
            pos:self.pos
            size:self.size
        Color:
            rgba:1,1,1,1
        Line:
            points:[self.center_x,0,self.center_x,self.height]
            width:2
    PongBall:
        id: pong_ball
        center:self.parent.center
    PongPaddle:
        id: left_paddle
        x: root.x+20
        center_y: root.center_y
    PongPaddle:
        id: right_paddle
        x: root.width - self.width -20
        center_y: root.center_y
    Label:
        text: str(root.player1.score)
        font_size:40
        center_x: root.width/4
        top: root.top -20
        color:1,1,1,1
    Label:
        text: str(root.player2.score)
        font_size:40
        center_x: root.width*3/4
        top: root.top -20
        color:1,1,1,1
""")
        game.serve_ball()
        Window.bind(on_key_down=lambda w, sc, cp, key, mods: game.keyboard_action(key))
        Clock.schedule_interval(game.update, 1/60.)
        return game

# --- Spiel starten ---
if __name__ == '__main__':
    PongApp().run()
