# ================== CONFIG ==================
from kivy.config import Config
Config.set('graphics', 'fullscreen', 'auto')
Config.set('graphics', 'resizable', '1')
Config.set('graphics', 'multisamples', '0')
Config.set('kivy', 'exit_on_escape', '0')
Config.set('input', 'mouse', 'mouse,disable_multitouch')

# ================== IMPORTS ==================
import random, os
from kivy.app import App
from kivy.clock import Clock
from kivy.uix.widget import Widget
from kivy.uix.screenmanager import ScreenManager, Screen
from kivy.uix.boxlayout import BoxLayout
from kivy.uix.label import Label
from kivy.uix.button import Button
from kivy.uix.textinput import TextInput
from kivy.graphics import Rectangle, Color
from kivy.core.window import Window
from kivy.lang import Builder
from kivy.metrics import sp

# ================== SCREEN / GRID ==================
Window.borderless = True
SCREEN_W, SCREEN_H = Window.system_size
TOP_UI = int(SCREEN_H * 0.1)

CELL_SIZE = int(sp(48))  # feste Blockgröße für scharfen Text

COLUMNS = SCREEN_W // CELL_SIZE
ROWS = (SCREEN_H - TOP_UI) // CELL_SIZE

# Sicherheitsgrenzen
COLUMNS = max(8, min(COLUMNS, 24))
ROWS = max(16, min(ROWS, 32))

Builder.load_file("settings.kv")
SCORE_FILE = "scores.txt"

# ================== TETROMINOS ==================
TETROMINOS = {
    'I': [[1,1,1,1]],
    'O': [[1,1],[1,1]],
    'T': [[0,1,0],[1,1,1]],
    'S': [[0,1,1],[1,1,0]],
    'Z': [[1,1,0],[0,1,1]],
    'J': [[1,0,0],[1,1,1]],
    'L': [[0,0,1],[1,1,1]],
}

# ================== LOGIN ==================
class LoginScreen(Screen):
    def __init__(self, **kw):
        super().__init__(**kw)
        box = BoxLayout(orientation="vertical", padding=40, spacing=20)
        box.add_widget(Label(text="TETRIS", font_size=sp(48)))
        self.user = TextInput(hint_text="Benutzername", multiline=False)
        self.status = Label(font_size=sp(18))
        btn = Button(text="Start", font_size=sp(22))
        btn.bind(on_press=self.login)
        box.add_widget(self.user)
        box.add_widget(self.status)
        box.add_widget(btn)
        self.add_widget(box)

    def login(self, *_):
        if not self.user.text:
            self.status.text = "Name eingeben"
            return
        game = self.manager.get_screen("game")
        game.set_user(self.user.text)
        self.manager.current = "game"

# ================== BOARD ==================
class TetrisBoard(Widget):
    def __init__(self, parent, **kw):
        super().__init__(**kw)
        self.parent_screen = parent
        self.grid = [[0]*COLUMNS for _ in range(ROWS)]
        self.pieces = []  # aktive Tetrominos (für Doppelstück)
        self.next_pieces = []  # Vorschau
        self.init_next_pieces()
        self.new_piece()

    def init_next_pieces(self):
        count = 2 if self.parent_screen.double_piece_mode else 1
        self.next_pieces = [random.choice(list(TETROMINOS.values())) for _ in range(count)]

    def new_piece(self):
        self.pieces = []
        count = 2 if self.parent_screen.double_piece_mode else 1
        for i in range(count):
            piece = self.next_pieces[i % len(self.next_pieces)]
            color = (random.random(), random.random(), random.random()) if self.parent_screen.random_colors else (1,1,1)
            x = max(0, COLUMNS//2 - len(piece[0])//2 + (-2 if i==0 else 2))
            y = 0
            self.pieces.append({"piece": piece, "color": color, "x": x, "y": y})
        # neue Vorschau generieren
        self.init_next_pieces()
        # Kollision prüfen
        for p in self.pieces:
            if self.collides_piece(p):
                self.parent_screen.reset_game()

    def collides_piece(self, p, dx=0, dy=0):
        for y,r in enumerate(p["piece"]):
            for x,c in enumerate(r):
                if c:
                    nx, ny = p["x"] + x + dx, p["y"] + y + dy
                    if nx < 0 or nx >= COLUMNS or ny >= ROWS:
                        return True
                    if ny >= 0 and self.grid[ny][nx]:
                        return True
        return False

    def collides(self, dx=0, dy=0):
        return any(self.collides_piece(p, dx, dy) for p in self.pieces)

    def lock(self):
        for p in self.pieces:
            for y,r in enumerate(p["piece"]):
                for x,c in enumerate(r):
                    if c:
                        self.grid[p["y"]+y][p["x"]+x] = p["color"]
        self.clear_lines()
        self.new_piece()
        # Geschwindigkeit anpassen
        self.parent_screen.speed = max(0.025, 0.5 * 0.95**(self.parent_screen.level-1))
        Clock.unschedule(self.parent_screen.update)
        Clock.schedule_interval(self.parent_screen.update, self.parent_screen.speed)

    def clear_lines(self):
        new = [r for r in self.grid if any(c==0 for c in r)]
        lines = ROWS - len(new)
        for _ in range(lines):
            new.insert(0,[0]*COLUMNS)

        if lines:
            # Level steigt sofort pro Linie
            self.parent_screen.level += lines
            score = lines * 100 * self.parent_screen.level
            if self.parent_screen.double_score and self.parent_screen.double_score_active:
                score *= 2
                self.parent_screen.double_score_active = False  # nur EINMAL pro Aktivierung

            self.parent_screen.score += score
            self.parent_screen.update_labels()

        self.grid = new

    def draw(self):
        self.canvas.clear()
        offset_x = (SCREEN_W - COLUMNS*CELL_SIZE)//2
        offset_y = 0
        mini_size = CELL_SIZE//2
        with self.canvas:
            # ================== BOARD RAHMEN ==================
            board_w = COLUMNS * CELL_SIZE
            board_h = ROWS * CELL_SIZE
            bx = offset_x
            by = offset_y

            # linker Rand (Glow Effekt)
            Color(0.2, 0.8, 1, 0.8)
            Rectangle(pos=(bx - 8, by), size=(4, board_h))
            Color(0.2, 0.8, 1, 0.4)
            Rectangle(pos=(bx - 12, by), size=(4, board_h))

            # rechter Rand
            Color(1, 0.3, 0.6, 0.8)
            Rectangle(pos=(bx + board_w + 4, by), size=(4, board_h))
            Color(1, 0.3, 0.6, 0.4)
            Rectangle(pos=(bx + board_w + 8, by), size=(4, board_h))

            # obere Linie
            Color(1, 1, 1, 0.6)
            Rectangle(pos=(bx - 12, by + board_h), size=(board_w + 24, 2))

            # untere Linie
            Color(1, 1, 1, 0.2)
            Rectangle(pos=(bx - 12, by - 2), size=(board_w + 24, 2))

            # ================== BOARD INHALT ================== 
        
            for y in range(ROWS):
                for x in range(COLUMNS):
                    if self.grid[y][x]:
                        Color(*self.grid[y][x])
                        Rectangle(
                            pos=(round(offset_x + x*CELL_SIZE),
                                 round(offset_y + (ROWS-y-1)*CELL_SIZE)),
                            size=(CELL_SIZE-1, CELL_SIZE-1)
                        )
            # Aktive Tetrominos
            for p in self.pieces:
                for y,r in enumerate(p["piece"]):
                    for x,c in enumerate(r):
                        if c:
                            Color(*p["color"])
                            Rectangle(
                                pos=(round(offset_x + (p["x"]+x)*CELL_SIZE),
                                     round(offset_y + (ROWS-(p["y"]+y)-1)*CELL_SIZE)),
                                size=(CELL_SIZE-1, CELL_SIZE-1)
                            )
            # Vorschau zeichnen
            for i, piece in enumerate(self.next_pieces):
                px = SCREEN_W - 6*mini_size
                py = SCREEN_H - TOP_UI - (i+1)*5*mini_size
                for y,r in enumerate(piece):
                    for x,c in enumerate(r):
                        if c:
                            Color(1,1,1)
                            Rectangle(
                                pos=(px + x*mini_size, py - y*mini_size),
                                size=(mini_size-1, mini_size-1)
                            )

# ================== GAME ==================
class TetrisGame(Screen):
    def __init__(self, **kw):
        super().__init__(**kw)
        self.username = ""
        self.score = 0
        self.level = 1
        self.speed = 0.5
        self.random_colors = True
        self.double_piece_mode = False
        self.double_score_active = False  # Trackt, ob das Power-Up gerade benutzt wurde

        # POWER UPS
        self.double_score = False
        self.double_time = 0
        self.joker_used = False
        self.sort_cd = 0

        top = BoxLayout(size_hint=(1,None), height=TOP_UI)
        self.l_user = Label(font_size=sp(20))
        self.l_score = Label(font_size=sp(20))
        btn = Button(text="⚙", font_size=sp(22), size_hint_x=0.2)
        btn.bind(on_press=lambda x:self.manager.switch_to_settings())
        self.btn_double = Button(text="1️⃣ Double\nREADY", font_size=sp(14))
        self.btn_joker  = Button(text="2️⃣ Joker\nREADY", font_size=sp(14))
        self.btn_sort   = Button(text="3️⃣ Sort\nREADY", font_size=sp(14))

        self.btn_double.bind(on_press=lambda x: self.activate_double())
        self.btn_joker.bind(on_press=lambda x: self.activate_joker())
        self.btn_sort.bind(on_press=lambda x: self.activate_sort())

        for w in (
            self.l_user,
            self.l_score,
            self.btn_double,
            self.btn_joker,
            self.btn_sort,
            btn
        ):
            top.add_widget(w)

        self.board = TetrisBoard(self)
        box = BoxLayout(orientation="vertical")
        box.add_widget(top)
        box.add_widget(self.board)
        self.add_widget(box)

        self.kb = Window.request_keyboard(None,self)
        self.kb.bind(on_key_down=self.key)
        Clock.schedule_interval(self.update, self.speed)

    # ---------- SETTINGS ----------
    def set_speed(self, s):
        self.speed = s
        Clock.unschedule(self.update)
        Clock.schedule_interval(self.update, self.speed)

    def toggle_colors(self):
        self.random_colors = not self.random_colors

    def double_piece_button(self):
        self.double_piece_mode = True

    def single_piece_button(self):
        """Schaltet zurück auf Einzel-Tetraminos"""
        self.double_piece_mode = False

    # ---------- POWER UPS ----------
    def activate_double(self):
        if self.double_time > 0:
            return  # Cooldown läuft
        self.double_score = True        # aktiviert das Power-Up
        self.double_score_active = True # dieses Flag sagt: noch aktiv
        self.double_time = 60           # Cooldown-Zähler

    def activate_joker(self):
        if self.joker_used: return
        self.joker_used = True
        self.board.grid = [[0]*COLUMNS for _ in range(ROWS)]

    def activate_sort(self):
        if self.sort_cd > 0: return
        blocks = [c for r in self.board.grid for c in r if c]
        self.board.grid = [[0]*COLUMNS for _ in range(ROWS)]
        x,y = 0,ROWS-1
        for c in blocks:
            self.board.grid[y][x] = c
            x+=1
            if x>=COLUMNS:
                x=0; y-=1
        self.sort_cd = 30

    # ---------- INPUT ----------
    def key(self, kb, keycode, text, mods):
        k = keycode[1]
        for p in self.board.pieces:
            if k=="left" and not self.board.collides(dx=-1): p["x"]-=1
            elif k=="right" and not self.board.collides(dx=1): p["x"]+=1
            elif k=="down" and not self.board.collides(dy=1): p["y"]+=1
            elif k=="up":
                old = [row[:] for row in p["piece"]]
                p["piece"] = [list(r) for r in zip(*old[::-1])]
                if self.board.collides(): p["piece"] = old
        if k=="1": self.activate_double()
        elif k=="2": self.activate_joker()
        elif k=="3": self.activate_sort()
        self.board.draw()

    def update(self, dt):
        # ---------- DOUBLE SCORE ----------
        if self.double_time > 0:
            self.double_time -= dt
            self.btn_double.text = f"1️⃣ Double\n{int(self.double_time)}s"
        else:
            self.double_score = False
            self.btn_double.text = "1️⃣ Double\nREADY"

        # ---------- JOKER ----------
        self.btn_joker.text = "2️⃣ Joker\nUSED" if self.joker_used else "2️⃣ Joker\nREADY"

        # ---------- SORT ----------
        if self.sort_cd > 0:
            self.sort_cd -= dt
            self.btn_sort.text = f"3️⃣ Sort\n{int(self.sort_cd)}s"
        else:
            self.btn_sort.text = "3️⃣ Sort\nREADY"
        if not self.board.collides(dy=1):
           for p in self.board.pieces:
               p["y"] += 1
        else:
            self.board.lock()

        self.board.draw()

    

    def reset_game(self):
        self.score = 0
        self.level = 1
        self.double_score = False
        self.joker_used = False
        self.double_piece_mode = False
        self.board.grid = [[0]*COLUMNS for _ in range(ROWS)]
        self.board.init_next_pieces()
        self.board.new_piece()
        self.update_labels()
        self.double_time = 0
        self.sort_cd = 0
        self.joker_used = False

    def set_user(self,u):
        self.username = u
        self.update_labels()

    def update_labels(self):
        self.l_user.text = f"User: {self.username}"
        self.l_score.text = f"Score: {self.score} | Level: {self.level}"

# ================== SETTINGS ==================
class SettingsScreen(Screen): pass

class Manager(ScreenManager):
    def switch_to_settings(self): self.current="settings"
    def switch_to_game(self): self.current="game"

class TetrisApp(App):
    def build(self):
        sm = Manager()
        sm.add_widget(LoginScreen(name="login"))
        sm.add_widget(TetrisGame(name="game"))
        sm.add_widget(SettingsScreen(name="settings"))
        return sm

if __name__=="__main__":
    TetrisApp().run()